"use strict";

const { startOfMonth, endOfMonth, subMonths, format } = require("date-fns");
// const { formatDate } = require("../helpers/dateHelper");
const BookingModel = require("../models/bookingModel");

const PropertyModel = require("../models/propertyModel");
const Property = new PropertyModel();
const Booking = new BookingModel();

const formatDate = format;
class RevenueService {
  async getMonthlyBreakup(revDate) {
    const data = {};
    const current = await this.getRevenueBreakup(
      startOfMonth(revDate),
      endOfMonth(revDate),
    );

    current.forEach((c) => {
      c.date = formatDate(c.end, "dd");
    });

    data.current = current;
    const previous = await this.getRevenueBreakup(
      subMonths(startOfMonth(revDate), 1),
      subMonths(endOfMonth(revDate), 1),
    );

    previous.forEach((p) => {
      p.date = formatDate(p.end, "EEE");
    });

    data.previous = previous;
    data.total = "10000";
    data.previousTotal = "11000";

    return data;
  }
  async earningsByMonth(property_id) {
    const now = new Date();
    const currentYear = now.getFullYear();
    const currentMonth = now.getMonth() + 1;

    let financialYearStart;
    if (currentMonth >= 4) {
      financialYearStart = `${currentYear}-04-01`;
    } else {
      financialYearStart = `${currentYear - 1}-04-01`;
    }

    const start = financialYearStart;
    const end = format(endOfMonth(new Date()), "yyyy-MM-dd");

    const { earnings, nights } = await Property.earningsByMonth(
      property_id,
      start,
      end,
    );

    return { earnings, nights };
  }
  async getMonthlyReportData(propertyId, month, year) {
    const startDate = `${year}-${month.toString().padStart(2, "0")}-01`;
    const endDate = format(endOfMonth(new Date(year, month - 1)), "yyyy-MM-dd");

    // Property details
    const propertyDetails = await Property.find("properties", {
      id: propertyId,
    });

    // Monthly bookings
    const bookings = await Booking.getMonthlyBookings(
      propertyId,
      startDate,
      endDate,
    );

    // Revenue data
    const revenueData = await Booking.getMonthlyRevenue(
      propertyId,
      startDate,
      endDate,
    );

    // Metrics
    const totalBookings = bookings.length;

    const totalRevenue = revenueData.reduce(
      (sum, record) => sum + (record.revenue || 0),
      0,
    );

    const averageDailyRate =
      totalBookings > 0 ? Math.round(totalRevenue / totalBookings) : 0;

    const daysInMonth = new Date(year, month, 0).getDate();

    const bookedNights = bookings.reduce((sum, booking) => {
      const checkIn = new Date(booking.start);
      const checkOut = new Date(booking.end);
      const nights = Math.ceil((checkOut - checkIn) / (1000 * 60 * 60 * 24));
      return sum + nights;
    }, 0);

    const occupancyRate = Math.round((bookedNights / daysInMonth) * 100);

    return {
      property: propertyDetails,
      period: {
        month,
        year,
        startDate,
        endDate,
      },
      summary: {
        totalBookings,
        totalRevenue: Math.round(totalRevenue),
        averageDailyRate,
        occupancyRate,
        bookedNights,
        daysInMonth,
      },
      bookings,
      revenueData,
      generatedAt: new Date(),
    };
  }
}

module.exports = new RevenueService();
