const express = require('express')
const router = express.Router()

const authenticateUser = require("../middleware/userAuthenticationHandler");
const EzeeHelper = require('../helpers/ezeeHelper');
const ezeeHelper = new EzeeHelper(process.env.EZEE_URL);

const bookingsController = require('../controllers/bookingsController');
const BookingsController = new bookingsController(ezeeHelper);
const dashboardController = require('../controllers/dashboardController');
const DashboardController = new dashboardController();
const usersController = require('../controllers/usersController');
const UsersController = new usersController();
const serviceRequestController = require('../controllers/serviceRequestController');
const ServiceRequestController = new serviceRequestController();
const settingsController = require('../controllers/settingsController');
const SettingsController = new settingsController();

const hostsController = require('../controllers/hostsController');
const HostsController = new hostsController();
const formsController = require('../controllers/formsController');
const FormsController = new formsController();
const propertyController = require('../controllers/propertyController');
const PropertyController = new propertyController();
const availabilityController = require('../controllers/availabilityController');
const AvailabilityController = new availabilityController(ezeeHelper);
const statementsController = require('../controllers/statementsController');
const StatementsController = new statementsController();
const liveBookingController = require('../controllers/liveBookingController');
const LiveBookingController = new liveBookingController();


const multer = require('multer');
const storage = multer.memoryStorage();
const upload = multer({ 
    storage: storage,
    limits: {
        fileSize: 5 * 1024 * 1024 // 5MB limit
    }
});

/* login and logout*/
router.get('/login/:flash?',(req,res)=>{
    const flash = req.params.flash || '';
    res.render('login',{flash:flash});
});
router.post("/handleLogin", upload.any(), UsersController.loginUser);
router.get('/logout',(req,res)=>{
    res.clearCookie('authcookie');
    res.render('login',{flash:''});
});
router.get('/forgotPassword',(req,res)=>{
    res.render('forgotPassword');
});
router.get('/check',BookingsController.check.bind(BookingsController));

/* admin Dashboard routes */
router.get('/',authenticateUser,DashboardController.index.bind(DashboardController));
router.get('/me',authenticateUser,UsersController.me.bind(UsersController));
router.post('/chart/',upload.any(),authenticateUser,DashboardController.updateChart.bind(DashboardController));
router.get('/revenues',authenticateUser,DashboardController.revenues.bind(DashboardController));
// New route for stacked earnings chart data
router.get('/stacked-earnings',authenticateUser,DashboardController.stackedEarningsData.bind(DashboardController));
router.get("/checkUnique/:type/:value/:userType/:currentId?",authenticateUser,UsersController.checkUnique.bind(UsersController));

/* admin booking routes */
router.get("/bookings/",authenticateUser,BookingsController.bookingsList);
router.get("/manageBookings/",authenticateUser,BookingsController.manageBookingsList);
router.get("/bookings/edit/:id",authenticateUser,BookingsController.bookingDetails);
router.get("/bookings/adjust/:id",authenticateUser,BookingsController.adjustPrices);

/* admin live booking information routes */
router.get("/liveBookingInfo/",authenticateUser,LiveBookingController.index.bind(LiveBookingController));
router.get("/api/liveBookingInfo/bookings",authenticateUser,LiveBookingController.getBookings.bind(LiveBookingController));
router.get("/api/liveBookingInfo/statistics",authenticateUser,LiveBookingController.getStatistics.bind(LiveBookingController));
router.post("/api/liveBookingInfo/sync",authenticateUser,LiveBookingController.manualSync.bind(LiveBookingController));

/* admin calendar routes */
router.get('/bookingsCalendar',authenticateUser, async function(...args){
    const results = await PropertyController.basicList(...args);
    return args[0].res.render('reservations/calendar.ejs',{properties:results});
});
router.get('/propertyCalendar/',authenticateUser, AvailabilityController.datesByProperty.bind(AvailabilityController));
router.post('/block',authenticateUser,function(...args){
    return BookingsController.block(...args);
});
router.post('/unblock',authenticateUser, function(...args){
    return BookingsController.unblock(...args);
});

/* admin guest routes */
router.get("/guests/",authenticateUser,UsersController.guests.bind(UsersController));
router.get("/guests/edit/:id",authenticateUser,UsersController.userDetails.bind(UsersController));

/* admin staff and hosts routes */
router.get("/users/:type/:flash?",authenticateUser,UsersController.users.bind(UsersController));
router.get("/user/add/:type",authenticateUser,UsersController.add.bind(UsersController));
router.post("/user/create",upload.any(),authenticateUser,UsersController.create.bind(UsersController));
router.get("/user/edit/:id",authenticateUser,UsersController.edit.bind(UsersController));
router.post("/user/save",upload.any(),authenticateUser,UsersController.save.bind(UsersController));
router.get("/changePassword/:id?",authenticateUser,UsersController.changePassword.bind(UsersController));
router.post("/savePassword",upload.any(),authenticateUser,UsersController.savePassword.bind(UsersController));
router.post('/generateOTP',upload.any(), UsersController.generateUserOTP.bind(UsersController));
router.post("/resetPassword",upload.any(),UsersController.resetPassword.bind(UsersController));

/* admin service requests routes */
router.get("/services/:flash?",authenticateUser,ServiceRequestController.index.bind(ServiceRequestController));
router.get("/service/check",ServiceRequestController.check.bind(ServiceRequestController));
router.get("/service/add",authenticateUser,ServiceRequestController.add.bind(ServiceRequestController));
router.post("/service/create", upload.any(),authenticateUser,ServiceRequestController.create.bind(ServiceRequestController));
router.get("/service/edit/:id",authenticateUser,ServiceRequestController.edit.bind(ServiceRequestController));
router.post("/service/save", upload.any(),authenticateUser,ServiceRequestController.save.bind(ServiceRequestController));
router.post("/service/updateStatus", upload.any(),authenticateUser,ServiceRequestController.updateStatus.bind(ServiceRequestController));
router.post("/service/delete-image",authenticateUser,ServiceRequestController.deleteImage);
router.post("/service/upload-image",upload.any(),authenticateUser,ServiceRequestController.uploadImage.bind(ServiceRequestController));

router.get("/forms/:flash?",authenticateUser,FormsController.index.bind(FormsController));

/* admin settings routes */
router.get("/settings/:flash?",authenticateUser,SettingsController.index.bind(SettingsController));
router.post("/updateDiscount/",upload.any(),authenticateUser,SettingsController.updateDiscount.bind(SettingsController));

/* admin accounting routes */
router.get("/statements/:property_id/:flash?",authenticateUser,StatementsController.statementsForProperty.bind(StatementsController));
router.post("/uploadStatement",upload.any(),authenticateUser,StatementsController.uploadStatement.bind(StatementsController));

/* Ezee IDs */
router.get('/channel_ids',authenticateUser,function(...args){
    return BookingsController.channelIds(...args);
});

router.get("/hosts/:propertyId", authenticateUser, HostsController.hostsByProperty.bind(HostsController));
router.get('/importBooking/:id', BookingsController.importBooking.bind(BookingsController));
router.get('/importBookingBulk', BookingsController.importBookingBulk.bind(BookingsController));

router.get("/roomsInfo",PropertyController.roomsInfo.bind(PropertyController));

router.get('/checkEmail',UsersController.checkEmail.bind(UsersController));
module.exports = router;