/**
 * Input validation rules for API endpoints using express-validator
 */
const { body, param, validationResult } = require('express-validator');

/**
 * Middleware to check for validation errors and return 400 if any
 */
const validate = (req, res, next) => {
    const errors = validationResult(req);
    if (!errors.isEmpty()) {
        return res.status(400).json({
            success: false,
            error: 'VALIDATION_ERROR',
            message: errors.array().map(e => e.msg).join(', '),
            details: errors.array()
        });
    }
    next();
};

/**
 * Booking creation validation
 */
const createBookingRules = [
    body('property_id').notEmpty().withMessage('Property ID is required').isInt().withMessage('Property ID must be a number'),
    body('start').notEmpty().withMessage('Check-in date is required').isISO8601().withMessage('Invalid check-in date format'),
    body('end').notEmpty().withMessage('Check-out date is required').isISO8601().withMessage('Invalid check-out date format'),
    validate
];

/**
 * OTP generation validation
 */
const generateOTPRules = [
    body('phone')
        .optional()
        .isMobilePhone('any')
        .withMessage('Invalid phone number format'),
    body('email')
        .optional()
        .isEmail()
        .withMessage('Invalid email format'),
    body()
        .custom((value, { req }) => {
            if (!req.body.phone && !req.body.email) {
                throw new Error('Phone number or email is required');
            }
            return true;
        }),
    validate
];

/**
 * Login with OTP validation
 */
const loginWithOTPRules = [
    body('phone').notEmpty().withMessage('Phone number is required'),
    body('otp').notEmpty().withMessage('OTP is required')
        .isLength({ min: 4, max: 6 }).withMessage('OTP must be 4-6 digits'),
    validate
];

/**
 * Contact form submission validation
 */
const submitFormRules = [
    body('form_type').notEmpty().withMessage('Form type is required')
        .isIn(['get_in_touch', 'contact_us', 'host_page_form']).withMessage('Invalid form type'),
    body('name').notEmpty().withMessage('Name is required')
        .trim()
        .isLength({ max: 100 }).withMessage('Name must be under 100 characters'),
    body('email').notEmpty().withMessage('Email is required')
        .isEmail().withMessage('Invalid email format')
        .normalizeEmail(),
    body('phone').notEmpty().withMessage('Phone is required'),
    validate
];

/**
 * Host enquiry validation
 */
const hostEnquiryRules = [
    body('location').notEmpty().withMessage('Location is required'),
    body('property_type').notEmpty().withMessage('Property type is required'),
    body('bedrooms').notEmpty().withMessage('Bedrooms count is required'),
    body('pool_type').notEmpty().withMessage('Pool type is required'),
    body('name').notEmpty().withMessage('Name is required').trim(),
    body('email').notEmpty().withMessage('Email is required').isEmail().withMessage('Invalid email'),
    body('otp').notEmpty().withMessage('OTP is required'),
    validate
];

/**
 * Profile update validation
 */
const updateProfileRules = [
    body('firstname').optional().trim().isLength({ max: 50 }).withMessage('First name must be under 50 characters'),
    body('lastname').optional().trim().isLength({ max: 50 }).withMessage('Last name must be under 50 characters'),
    body('email').optional().isEmail().withMessage('Invalid email format').normalizeEmail(),
    validate
];

/**
 * Property creation validation
 */
const createPropertyRules = [
    body('name').optional().trim().isLength({ max: 200 }).withMessage('Property name must be under 200 characters'),
    validate
];

/**
 * Collection CRUD validation
 */
const createCollectionRules = [
    body('name').notEmpty().withMessage('Collection name is required').trim(),
    validate
];

/**
 * Destination CRUD validation
 */
const createDestinationRules = [
    body('name').notEmpty().withMessage('Destination name is required').trim(),
    validate
];

/**
 * Blog CRUD validation
 */
const createBlogRules = [
    body('title').notEmpty().withMessage('Blog title is required').trim(),
    validate
];

/**
 * ID parameter validation
 */
const idParamRules = [
    param('id').isInt({ min: 1 }).withMessage('Invalid ID parameter'),
    validate
];

module.exports = {
    validate,
    createBookingRules,
    generateOTPRules,
    loginWithOTPRules,
    submitFormRules,
    hostEnquiryRules,
    updateProfileRules,
    createPropertyRules,
    createCollectionRules,
    createDestinationRules,
    createBlogRules,
    idParamRules
};
